import { RETAINED_BY, RETAIN_METHOD, RELEASE_METHOD } from './types.mjs';
export { RELEASE_METHOD, RETAINED_BY, RETAIN_METHOD } from './types.mjs';

class StackFrame {
  constructor() {
    this.memoryManaged = new Set();
  }

  add(memoryManageable) {
    this.memoryManaged.add(memoryManageable);
    memoryManageable[RETAINED_BY].add(this);
    memoryManageable[RETAIN_METHOD]();
  }

  release() {
    for (const memoryManaged of this.memoryManaged) {
      memoryManaged[RETAINED_BY].delete(this);
      memoryManaged[RELEASE_METHOD]();
    }

    this.memoryManaged.clear();
  }

}
function isMemoryManageable(value) {
  return Boolean(value && value[RETAIN_METHOD] && value[RELEASE_METHOD]);
}
function retain(value, {
  deep = true
} = {}) {
  return retainInternal(value, deep, new Map());
}

function retainInternal(value, deep, seen) {
  const seenValue = seen.get(value);
  if (seenValue != null) return seenValue;
  const canRetain = isMemoryManageable(value);

  if (canRetain) {
    value[RETAIN_METHOD]();
  }

  seen.set(value, canRetain);

  if (deep) {
    if (Array.isArray(value)) {
      const nestedCanRetain = value.reduce((canRetain, item) => retainInternal(item, deep, seen) || canRetain, canRetain);
      seen.set(value, nestedCanRetain);
      return nestedCanRetain;
    }

    if (isBasicObject(value)) {
      const nestedCanRetain = Object.keys(value).reduce((canRetain, key) => retainInternal(value[key], deep, seen) || canRetain, canRetain);
      seen.set(value, nestedCanRetain);
      return nestedCanRetain;
    }
  }

  seen.set(value, canRetain);
  return canRetain;
}

function release(value, {
  deep = true
} = {}) {
  return releaseInternal(value, deep, new Map());
}
function releaseInternal(value, deep, seen) {
  const seenValue = seen.get(value);
  if (seenValue != null) return seenValue;
  const canRelease = isMemoryManageable(value);

  if (canRelease) {
    value[RELEASE_METHOD]();
  }

  seen.set(value, canRelease);

  if (deep) {
    if (Array.isArray(value)) {
      const nestedCanRelease = value.reduce((canRelease, item) => releaseInternal(item, deep, seen) || canRelease, canRelease);
      seen.set(value, nestedCanRelease);
      return nestedCanRelease;
    }

    if (isBasicObject(value)) {
      const nestedCanRelease = Object.keys(value).reduce((canRelease, key) => releaseInternal(value[key], deep, seen) || canRelease, canRelease);
      seen.set(value, nestedCanRelease);
      return nestedCanRelease;
    }
  }

  return canRelease;
}
function isBasicObject(value) {
  if (value == null || typeof value !== 'object') return false;
  const prototype = Object.getPrototypeOf(value);
  return prototype == null || prototype === Object.prototype;
}

export { StackFrame, isBasicObject, isMemoryManageable, release, releaseInternal, retain };
