import { BaseFunctionConfigurationSchema } from './schemas.js';
import { blocks, defaultFunctionsFlavors, withJavaScriptFunctionsFlavors } from '../../constants.js';
import { constantize } from '@shopify/cli-kit/common/string';
import { partnersFqdn } from '@shopify/cli-kit/node/context/fqdn';
import { areJavaScriptFunctionsEnabled } from '@shopify/cli-kit/node/context/local';
import { joinPath, basename } from '@shopify/cli-kit/node/path';
/**
 * Class that represents an instance of a local function
 * Before creating this class we've validated that:
 * - There is a spec for this type of function
 * - The Config Schema for that spec is followed by the function config toml file
 *
 * This class holds the public interface to interact with functions
 */
export class FunctionInstance {
    constructor(options) {
        this.configuration = options.configuration;
        this.configurationPath = options.configurationPath;
        this.specification = options.specification;
        this.directory = options.directory;
        this.entrySourceFilePath = options.entryPath;
        this.localIdentifier = basename(options.directory);
        this.idEnvironmentVariableName = `SHOPIFY_${constantize(basename(this.directory))}_ID`;
    }
    get graphQLType() {
        return this.specification.identifier.toUpperCase();
    }
    get identifier() {
        return this.specification.identifier;
    }
    get type() {
        return this.specification.identifier;
    }
    get externalType() {
        return this.specification.externalIdentifier;
    }
    get name() {
        return this.configuration.name;
    }
    get buildCommand() {
        return this.configuration.build.command;
    }
    get inputQueryPath() {
        return joinPath(this.directory, 'input.graphql');
    }
    get buildWasmPath() {
        const relativePath = this.configuration.build.path ?? joinPath('dist', 'index.wasm');
        return joinPath(this.directory, relativePath);
    }
    get isJavaScript() {
        return Boolean(this.entrySourceFilePath?.endsWith('.js') || this.entrySourceFilePath?.endsWith('.ts'));
    }
    async publishURL(options) {
        const fqdn = await partnersFqdn();
        return `https://${fqdn}/${options.orgId}/apps/${options.appId}/extensions`;
    }
}
/**
 * Create a new function spec.
 *
 * Everything but "identifer" and "templatePath" is optional.
 * ```ts
 * identifier: string // unique identifier for the function type
 * externalIdentifier: string // unique identifier used externally (default: same as "identifier")
 * externalName: string // human name used externally (default: same as "identifier")
 * helpURL?: string // URL to documentation
 * gated: boolean // whether the function is only accessible to shopifolk or not (default: false)
 * supportedFlavors: {name: string; value: string}[] // list of supported flavors (default: 'wasm' and 'rust')
 * configSchema: ZodSchemaType<TConfiguration> // schema for the function toml file (default: BaseFunctionConfigurationSchema)
 * registrationLimit: number // max number of functions of this type that can be registered (default: 10)
 * templateURL?: string // URL to the functions repository (default: 'https://github.com/Shopify/function-examples')
 * templatePath: (lang: string) => string // path to the template directory for the given language inside the templateURL repo
 * ```
 */
export function createFunctionSpecification(spec) {
    const defaults = {
        templateURL: 'https://github.com/Shopify/function-examples',
        externalIdentifier: spec.identifier,
        externalName: spec.identifier,
        supportedFlavors: areJavaScriptFunctionsEnabled() ? withJavaScriptFunctionsFlavors : defaultFunctionsFlavors,
        configSchema: BaseFunctionConfigurationSchema,
        gated: false,
        registrationLimit: spec.registrationLimit ?? blocks.functions.defaultRegistrationLimit,
        category: () => 'function',
    };
    return { ...defaults, ...spec };
}
//# sourceMappingURL=functions.js.map