import { groupBy, partition } from '@shopify/cli-kit/common/collection';
import { uniqBy, difference } from '@shopify/cli-kit/common/array';
import { pickBy } from '@shopify/cli-kit/common/object';
import { slugify } from '@shopify/cli-kit/common/string';
/**
 * Filter function to match a local and a remote source by type and name
 */
const sameTypeAndName = (local, remote) => {
    return remote.type === local.graphQLType && slugify(remote.title) === slugify(local.configuration.name);
};
/**
 * Automatically match local and remote sources if they have the same type and name
 *
 * If multiple local or remote sources have the same type and name, they can't be matched automatically
 */
function matchByNameAndType(local, remote, remoteIdField) {
    const uniqueLocal = uniqBy(local, (elem) => [elem.graphQLType, elem.configuration.name]);
    const uniqueRemote = uniqBy(remote, (elem) => [elem.type, elem.title]);
    const validMatches = {};
    uniqueLocal.forEach((localSource) => {
        const possibleMatch = uniqueRemote.find((remoteSource) => sameTypeAndName(localSource, remoteSource));
        if (possibleMatch)
            validMatches[localSource.localIdentifier] = possibleMatch[remoteIdField];
    });
    const pendingLocal = local.filter((elem) => !validMatches[elem.localIdentifier]);
    const pendingRemote = remote.filter((registration) => !Object.values(validMatches).includes(registration[remoteIdField]));
    return { matched: validMatches, pending: { local: pendingLocal, remote: pendingRemote } };
}
/**
 * Ask the user to confirm the relationship between a local source and a remote source if they
 * the only ones of their types.
 */
function matchByUniqueType(localSources, remoteSources) {
    const localGroups = groupBy(localSources, 'graphQLType');
    const localUnique = Object.values(pickBy(localGroups, (group, key) => group.length === 1)).flat();
    const remoteGroups = groupBy(remoteSources, 'type');
    const remoteUniqueMap = pickBy(remoteGroups, (group, key) => group.length === 1);
    const toConfirm = [];
    const toCreate = [];
    // for every local source that has a unique type we either:
    // - find a corresponding unique remote source and ask the user to confirm
    // - create it from scratch
    for (const local of localUnique) {
        const remote = remoteUniqueMap[local.graphQLType];
        if (remote && remote[0]) {
            toConfirm.push({ local, remote: remote[0] });
        }
        else {
            toCreate.push(local);
        }
    }
    // now for every local source with a duplicated type we check
    // if there is a remote source with the same type. if the answer is no,
    // it means that we need to create them.
    const localDuplicated = difference(localSources, localUnique);
    const remotePending = difference(remoteSources, toConfirm.map((elem) => elem.remote));
    const [localPending, localToCreate] = partition(localDuplicated, (local) => remotePending.map((remote) => remote.type).includes(local.graphQLType));
    toCreate.push(...localToCreate);
    return {
        toCreate,
        toConfirm,
        pending: {
            local: localPending,
            remote: remotePending,
        },
    };
}
/**
 * Automatically match local sources to remote sources.
 * If we can't match a local source to any remote sources, we can create it.
 * If we are unsure about the matching we can ask the user to confirm the relationship.
 */
export async function automaticMatchmaking(localSources, remoteSources, identifiers, remoteIdField) {
    const ids = getExtensionIds(localSources, identifiers);
    const localUUIDs = Object.values(ids);
    const existsRemotely = (local) => remoteSources.some((remote) => remote[remoteIdField] === ids[local.localIdentifier] && remote.type === local.graphQLType);
    // We try to automatically match sources if they have the same name and type,
    // by considering local sources which are missing on the remote side and
    // remote sources which are not synchronized locally.
    const { matched: matchedByNameAndType, pending: matchResult } = matchByNameAndType(localSources.filter((local) => !existsRemotely(local)), remoteSources.filter((remote) => !localUUIDs.includes(remote[remoteIdField])), remoteIdField);
    // Now we try to find a match between a local source and remote one if they have
    // the same type and they are unique even if they have different names. For example:
    // LOCAL_CHECKOUT_UI_NAMED_APPLE -> REMOTE_CHECKOUT_UI_NAMED_PEAR
    // LOCAL_PROD_SUBSCR_NAMED_ORANGE -> REMOTE_PROD_SUBSCR_NAMED_LEMON
    const { toConfirm, toCreate, pending } = matchByUniqueType(matchResult.local, matchResult.remote);
    return {
        identifiers: { ...ids, ...matchedByNameAndType },
        toConfirm,
        toCreate,
        toManualMatch: pending,
    };
}
export function getExtensionIds(localSources, identifiers) {
    const localSourcesIds = localSources.map((source) => source.localIdentifier);
    return pickBy(identifiers, (_, id) => localSourcesIds.includes(id));
}
//# sourceMappingURL=id-matching.js.map