import { renderConcurrent } from '@shopify/cli-kit/node/ui';
import { getAvailableTCPPort } from '@shopify/cli-kit/node/tcp';
import { AbortController } from '@shopify/cli-kit/node/abort';
import { outputDebug, outputContent, outputToken } from '@shopify/cli-kit/node/output';
import { openURL } from '@shopify/cli-kit/node/system';
import * as http from 'http';
/**
 * A convenient function that runs an HTTP server and does path-based traffic forwarding to sub-processes that run
 * an HTTP server. The method assigns a random port to each of the processes.
 * @param tunnelUrl - The URL of the tunnel.
 * @param portNumber - The port to use for the proxy HTTP server. When undefined, a random port is automatically assigned.
 * @param proxyTargets - List of target processes to forward traffic to.
 * @param additionalProcesses - Additional processes to run. The proxy won't forward traffic to these processes.
 * @returns A promise that resolves with an interface to get the port of the proxy and stop it.
 */
export async function runConcurrentHTTPProcessesAndPathForwardTraffic({ previewUrl, portNumber = undefined, proxyTargets, additionalProcesses, }) {
    // Lazy-importing it because it's CJS and we don't want it
    // to block the loading of the ESM module graph.
    const { default: httpProxy } = await import('http-proxy');
    const rules = {};
    const processes = await Promise.all(proxyTargets.map(async (target) => {
        const targetPort = await getAvailableTCPPort();
        rules[target.pathPrefix ?? '/'] = `http://localhost:${targetPort}`;
        return {
            prefix: target.logPrefix,
            action: async (stdout, stderr, signal) => {
                await target.action(stdout, stderr, signal, targetPort);
            },
        };
    }));
    const availablePort = portNumber ?? (await getAvailableTCPPort());
    outputDebug(outputContent `
Starting reverse HTTP proxy on port ${outputToken.raw(availablePort.toString())}
Routing traffic rules:
${outputToken.json(JSON.stringify(rules))}
`);
    const proxy = httpProxy.createProxy();
    const server = http.createServer(function (req, res) {
        const target = match(rules, req);
        if (target)
            return proxy.web(req, res, { target });
        outputDebug(`
Reverse HTTP proxy error - Invalid path: ${req.url}
These are the allowed paths:
${outputToken.json(JSON.stringify(rules))}
`);
        res.statusCode = 500;
        res.end(`Invalid path ${req.url}`);
    });
    // Capture websocket requests and forward them to the proxy
    server.on('upgrade', function (req, socket, head) {
        const target = match(rules, req);
        if (target)
            return proxy.ws(req, socket, head, { target });
        socket.destroy();
    });
    const abortController = new AbortController();
    abortController.signal.addEventListener('abort', () => {
        server.close();
    });
    let renderConcurrentOptions = {
        processes: [...processes, ...additionalProcesses],
        abortController,
    };
    if (previewUrl) {
        renderConcurrentOptions = {
            ...renderConcurrentOptions,
            onInput: (input, _key, exit) => {
                if (input === 'p' && previewUrl) {
                    // eslint-disable-next-line @typescript-eslint/no-floating-promises
                    openURL(previewUrl);
                }
                else if (input === 'q') {
                    exit();
                }
            },
            footer: {
                title: 'Press `p` to open your browser. Press `q` to quit.',
                subTitle: `Preview URL: ${previewUrl}`,
            },
        };
    }
    await Promise.all([renderConcurrent(renderConcurrentOptions), server.listen(availablePort)]);
}
function match(rules, req) {
    const path = req.url ?? '/';
    for (const pathPrefix in rules) {
        if (path.startsWith(pathPrefix))
            return rules[pathPrefix];
    }
    return undefined;
}
//# sourceMappingURL=http-reverse-proxy.js.map