import { buildHeaders, httpsAgent, RequestClientError, sanitizedHeadersOutput } from './headers.js';
import { stringifyMessage, outputContent, outputToken, outputDebug } from '../../../public/node/output.js';
import { AbortError } from '../../../public/node/error.js';
import { ClientError, GraphQLClient } from 'graphql-request';
import { performance } from 'perf_hooks';
export function graphqlRequest(query, api, url, token, variables, handleErrors = true) {
    const action = async () => {
        const headers = buildHeaders(token);
        debugLogRequest(api, query, variables, headers);
        const clientOptions = { agent: await httpsAgent(), headers };
        const client = new GraphQLClient(url, clientOptions);
        const t0 = performance.now();
        const response = await client.request(query, variables);
        const t1 = performance.now();
        outputDebug(`Request to ${url.toString()} completed in ${Math.round(t1 - t0)} ms`);
        return response;
    };
    if (handleErrors) {
        return handlingErrors(api, action);
    }
    else {
        return action();
    }
}
function debugLogRequest(api, query, variables, headers = {}) {
    outputDebug(outputContent `
Sending ${outputToken.json(api)} GraphQL request:
${outputToken.raw(query.toString())}

With variables:
${variables ? JSON.stringify(variables, null, 2) : ''}

And headers:
${sanitizedHeadersOutput(headers)}
`);
}
async function handlingErrors(api, action) {
    try {
        return await action();
    }
    catch (error) {
        if (error instanceof ClientError) {
            const errorMessage = stringifyMessage(outputContent `
  The ${outputToken.raw(api)} GraphQL API responded unsuccessfully with the HTTP status ${`${error.response.status}`} and errors:

  ${outputToken.json(error.response.errors)}
      `);
            let mappedError;
            if (error.response.status < 500) {
                mappedError = new RequestClientError(errorMessage, error.response.status);
            }
            else {
                mappedError = new AbortError(errorMessage);
            }
            mappedError.stack = error.stack;
            throw mappedError;
        }
        else {
            throw error;
        }
    }
}
//# sourceMappingURL=graphql.js.map