import { ZodSchemaType, BaseConfigContents } from './schemas.js';
import { ExtensionCategory, GenericSpecification, UIExtension } from '../app/extensions.js';
import { ExtensionFlavor } from '../../services/generate/extension.js';
import { Result } from '@shopify/cli-kit/node/result';
import { TokenizedString } from '@shopify/cli-kit/node/output';
/**
 * Extension specification with all the needed properties and methods to load an extension.
 */
export interface UIExtensionSpec<TConfiguration extends BaseConfigContents = BaseConfigContents> extends GenericSpecification {
    identifier: string;
    partnersWebIdentifier: string;
    surface: string;
    singleEntryPath: boolean;
    registrationLimit: number;
    supportedFlavors: {
        name: string;
        value: ExtensionFlavor;
    }[];
    gated: boolean;
    helpURL?: string;
    dependency?: {
        name: string;
        version: string;
    };
    templatePath?: string;
    graphQLType?: string;
    schema: ZodSchemaType<TConfiguration>;
    getBundleExtensionStdinContent?: (config: TConfiguration) => string;
    deployConfig?: (config: TConfiguration, directory: string) => Promise<{
        [key: string]: unknown;
    }>;
    validate?: (config: TConfiguration, directory: string) => Promise<Result<unknown, string>>;
    preDeployValidation?: (config: TConfiguration) => Promise<void>;
    category: () => ExtensionCategory;
    previewMessage?: (host: string, uuid: string, config: TConfiguration, storeFqdn: string) => TokenizedString | undefined;
    shouldFetchCartUrl?(config: TConfiguration): boolean;
    hasExtensionPointTarget?(config: TConfiguration, target: string): boolean;
}
/**
 * Class that represents an instance of a local extension
 * Before creating this class we've validated that:
 * - There is a spec for this type of extension
 * - The Schema for that spec is followed by the extension config toml file
 * - We were able to find an entry point file for that extension
 *
 * It supports extension points, making this Class compatible with both new ui-extension
 * and legacy extension types. Extension points are optional and this class will handle them if present.
 *
 * This class holds the public interface to interact with extensions
 */
export declare class UIExtensionInstance<TConfiguration extends BaseConfigContents = BaseConfigContents> implements UIExtension<TConfiguration> {
    entrySourceFilePath: string;
    outputBundlePath: string;
    devUUID: string;
    localIdentifier: string;
    idEnvironmentVariableName: string;
    directory: string;
    configuration: TConfiguration;
    configurationPath: string;
    private specification;
    get graphQLType(): string;
    get identifier(): string;
    get type(): string;
    get humanName(): string;
    get name(): string;
    get dependency(): {
        name: string;
        version: string;
    } | undefined;
    get externalType(): string;
    get surface(): string;
    constructor(options: {
        configuration: TConfiguration;
        configurationPath: string;
        entryPath: string;
        directory: string;
        specification: UIExtensionSpec;
    });
    deployConfig(): Promise<{
        [key: string]: unknown;
    }>;
    validate(): Promise<Result<unknown, string>>;
    preDeployValidation(): Promise<void>;
    publishURL(options: {
        orgId: string;
        appId: string;
        extensionId?: string;
    }): Promise<string>;
    previewMessage(url: string, storeFqdn: string): TokenizedString | undefined;
    getBundleExtensionStdinContent(): string;
    shouldFetchCartUrl(): boolean;
    hasExtensionPointTarget(target: string): boolean;
}
/**
 * These fields are forbidden when creating a new ExtensionSpec
 * They belong to the ExtensionSpec interface, but the values are obtained from the API
 * and should not be set by the user locally
 *
 * WARNING: 'surface' should be included here but is not yet compatible with the extension server
 */
export type ForbiddenFields = 'registrationLimit' | 'category' | 'externalIdentifier' | 'externalName' | 'name';
/**
 * Partial ExtensionSpec type used when creating a new ExtensionSpec, the only mandatory field is the identifier
 */
export interface CreateExtensionSpecType<TConfiguration extends BaseConfigContents = BaseConfigContents> extends Partial<Omit<UIExtensionSpec<TConfiguration>, ForbiddenFields>> {
    identifier: string;
}
/**
 * Create a new ui extension spec.
 *
 * Everything but "identifer" is optional.
 * ```ts
 * identifier: string // unique identifier for the extension type
 * externalIdentifier: string // identifier used externally (default: same as "identifier")
 * partnersWebIdentifier: string // identifier used in the partners web UI (default: same as "identifier")
 * surface?: string // surface where the extension is going to be rendered (default: 'unknown')
 * singleEntryPath: boolean // whether the extension has a single entry point (default: true)
 * supportedFlavors: {name: string; value: string}[] // list of supported flavors (default: 'javascript', 'typescript', 'typescript-react', 'javascript-react')
 * helpURL?: string // url to the help page for the extension, shown after generating the extension
 * dependency?: {name: string; version: string} // dependency to be added to the extension's package.json
 * templatePath?: string // path to the template to be used when generating the extension
 * graphQLType?: string // GraphQL type of the extension (default: same as "identifier")
 * schema?: ZodSchemaType<TConfiguration> // schema used to validate the extension's configuration (default: BaseUIExtensionSchema)
 * getBundleExtensionStdinContent?: (configuration: TConfiguration) => string // function to generate the content of the stdin file used to bundle the extension
 * validate?: (configuration: TConfiguration, directory: string) => Promise<Result<undefined, Error>> // function to validate the extension's configuration
 * preDeployValidation?: (configuration: TConfiguration) => Promise<void> // function to validate the extension's configuration before deploying it
 * deployConfig?: (configuration: TConfiguration, directory: string) => Promise<{[key: string]: unknown}> // function to generate the extensions configuration payload to be deployed
 * previewMessage?: (url: string, devUUID: string, configuration: TConfiguration, storeFqdn: string) => string | undefined // function to generate the preview message shown to the user during `dev`
 * shouldFetchCartUrl?: (configuration: TConfiguration) => boolean // function to determine if the extension should fetch the cart url
 * hasExtensionPointTarget?: (configuration: TConfiguration, target: string) => boolean // function to determine if the extension has a given extension point target
 * ```
 */
export declare function createUIExtensionSpecification<TConfiguration extends BaseConfigContents = BaseConfigContents>(spec: CreateExtensionSpecType<TConfiguration>): UIExtensionSpec<TConfiguration>;
