import { useRef, useEffect } from 'react';
import { useExtensionApi } from './api.mjs';
import { useSubscription } from './subscription.mjs';

/**
 * Returns the `buyerJourney` details on buyer progression in checkout.
 */
function useBuyerJourney() {
  return useExtensionApi().buyerJourney;
}

/**
 * Returns true if the buyer completed submitting their order.
 *
 * For example, when viewing the order status page after submitting payment, the buyer will have completed their order.
 */
function useBuyerJourneyCompleted() {
  const buyerJourney = useExtensionApi().buyerJourney;
  const buyerJourneyCompleted = useSubscription(buyerJourney.completed);
  return buyerJourneyCompleted;
}

/**
 * Returns a function to intercept and block navigation in checkout.
 */
function useBuyerJourneyIntercept(interceptor) {
  const buyerJourney = useExtensionApi().buyerJourney;
  const interceptorRef = useRef(interceptor);
  interceptorRef.current = interceptor;
  useEffect(() => {
    const teardownPromise = buyerJourney.intercept(interceptorProps => interceptorRef.current(interceptorProps));
    return () => {
      teardownPromise.then(teardown => teardown()).catch(() => {});
    };
  }, [buyerJourney]);
}

export { useBuyerJourney, useBuyerJourneyCompleted, useBuyerJourneyIntercept };
