import { FatalError as Fatal } from './error.js';
import { Logger, LogLevel } from './output.js';
import { ConcurrentOutputProps } from '../../private/node/ui/components/ConcurrentOutput.js';
import { AlertProps, CustomSection } from '../../private/node/ui/components/Alert.js';
import ScalarDict from '../../private/node/ui/components/Table/ScalarDict.js';
import { TableColumn, TableProps } from '../../private/node/ui/components/Table/Table.js';
import { SelectPromptProps } from '../../private/node/ui/components/SelectPrompt.js';
import { Task } from '../../private/node/ui/components/Tasks.js';
import { TextPromptProps } from '../../private/node/ui/components/TextPrompt.js';
import { AutocompletePromptProps } from '../../private/node/ui/components/AutocompletePrompt.js';
import { InlineToken, LinkToken, TokenItem } from '../../private/node/ui/components/TokenizedText.js';
import { Key as InkKey, RenderOptions } from 'ink';
type PartialBy<T, TKey extends keyof T> = Omit<T, TKey> & Partial<Pick<T, TKey>>;
export interface RenderConcurrentOptions extends PartialBy<ConcurrentOutputProps, 'abortController'> {
    renderOptions?: RenderOptions;
}
/**
 * Renders output from concurrent processes to the terminal with {@link ConcurrentOutput}.
 */
export declare function renderConcurrent({ renderOptions, ...props }: RenderConcurrentOptions): Promise<void>;
export type AlertCustomSection = CustomSection;
export type RenderAlertOptions = Omit<AlertProps, 'type'>;
/**
 * Renders an information banner to the console.
 *
 * Basic:
 *
 * ```
 * ╭─ info ───────────────────────────────────────────────────╮
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 *
 * Complete:
 * ```
 * ╭─ info ───────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * │  Next steps                                              │
 * │    • Run `cd santorini-goods`                            │
 * │    • To preview your project, run `npm app dev`          │
 * │    • To add extensions, run `npm generate extension`     │
 * │                                                          │
 * │  Reference                                               │
 * │    • Run `npm shopify help`                              │
 * │    • Press 'return' to open the dev docs:                │
 * │      https://shopify.dev                                 │
 * │                                                          │
 * │  Link: https://shopify.com                               │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export declare function renderInfo(options: RenderAlertOptions): void;
/**
 * Renders a success banner to the console.
 *
 * Basic:
 *
 * ```
 * ╭─ success ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 *
 * Complete:
 * ```
 * ╭─ success ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * │  Next steps                                              │
 * │    • Run `cd santorini-goods`                              │
 * │    • To preview your project, run `npm app dev`            │
 * │    • To add extensions, run `npm generate extension`       │
 * │                                                          │
 * │  Reference                                               │
 * │    • Run `npm shopify help`                                │
 * │    • Press 'return' to open the dev docs:                │
 * │      https://shopify.dev                                 │
 * │                                                          │
 * │  Link: https://shopify.com                               │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export declare function renderSuccess(options: RenderAlertOptions): void;
/**
 * Renders a warning banner to the console.
 *
 * Basic:
 *
 * ```
 * ╭─ warning ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 *
 * Complete:
 * ```
 * ╭─ warning ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * │  Next steps                                              │
 * │    • Run `cd santorini-goods`                            │
 * │    • To preview your project, run `npm app dev`          │
 * │    • To add extensions, run `npm generate extension`     │
 * │                                                          │
 * │  Reference                                               │
 * │    • Run `npm shopify help`                              │
 * │    • Press 'return' to open the dev docs:                │
 * │      https://shopify.dev                                 │
 * │                                                          │
 * │  Link: https://shopify.com                               │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export declare function renderWarning(options: RenderAlertOptions): void;
/**
 * Renders a Fatal error to the console inside a banner.
 *
 * ```
 * ╭─ error ──────────────────────────────────────────────────╮
 * │                                                          │
 * │  Couldn't connect to the Shopify Partner Dashboard.      │
 * │                                                          │
 * │  Check your internet connection and try again.           │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export declare function renderFatalError(error: Fatal): void;
/**
 * Renders a select prompt to the console.
 *
 * ```
 * ?  Associate your project with the org Castile Ventures?
 *
 *      Add:     • new-ext
 *
 *      Remove:  • integrated-demand-ext
 *               • order-discount
 *
 * \>  (f) first
 *     (s) second
 *     (3) third
 *     (4) fourth
 *     (5) seventh
 *     (6) tenth
 *
 *     Automations
 *     (7) fifth
 *     (8) sixth
 *
 *     Merchant Admin
 *     (9) eighth
 *     (10) ninth
 *
 *     Press ↑↓ arrows to select, enter to confirm
 * ```
 */
export declare function renderSelectPrompt<T>(props: Omit<SelectPromptProps<T>, 'onSubmit'>): Promise<T>;
export interface RenderConfirmationPromptOptions extends Pick<SelectPromptProps<boolean>, 'message' | 'infoTable'> {
    confirmationMessage?: string;
    cancellationMessage?: string;
}
/**
 * Renders a confirmation prompt to the console.
 *
 * ?  Do you want to continue?
 *
 * \>  (y) Yes, confirm
 *     (n) No, canccel
 *
 *     Press ↑↓ arrows to select, enter to confirm
 */
export declare function renderConfirmationPrompt({ message, infoTable, confirmationMessage, cancellationMessage, }: RenderConfirmationPromptOptions): Promise<boolean>;
/**
 * Renders an autocomplete prompt to the console.
 * ```
 * ?  Select a template  Type to search...

 * \>  first
 *     second
 *     third

 *  Press ↑↓ arrows to select, enter to confirm
 * ```
 */
export declare function renderAutocompletePrompt<T>(props: PartialBy<Omit<AutocompletePromptProps<T>, 'onSubmit'>, 'search'>): Promise<T>;
/**
 * Renders a table to the console.
 *
 * ```
 * name                      role           Identifier
 * ────────────────────────  ─────────────  ──────────
 * Dawn                      [live]         #1361
 * Studio                                   #1363
 * Debut                     [unpublished]  #1374
 * Development (1a23b4-MBP)  [development]  #1368
 * ```
 */
export declare function renderTable<T extends ScalarDict>(props: TableProps<T>): void;
/**
 * Runs async tasks and displays their progress to the console.
 */
export declare function renderTasks<TContext>(tasks: Task<TContext>[]): Promise<void>;
/**
 * Renders a text prompt to the console.
 * ```
 * ?  What is your name?
 * \>  John
 * ```
 */
export declare function renderTextPrompt(props: Omit<TextPromptProps, 'onSubmit'>): Promise<string>;
interface RenderTextOptions {
    text: string;
    logLevel?: LogLevel;
    logger?: Logger;
}
/** Renders a text string to the console.
 * Using this function makes sure that correct spacing is applied among the various components. */
export declare function renderText({ text, logLevel, logger }: RenderTextOptions): void;
/** Waits for any key to be pressed except Ctrl+C which will terminate the process. */
export declare const keypress: () => Promise<unknown>;
export type Key = InkKey;
export { Task, TokenItem, InlineToken, LinkToken, TableColumn };
