import { AbortSilentError } from './error.js';
import { collectLog, consoleError, consoleLog, outputDebug, outputWhereAppropriate } from './output.js';
import { isUnitTest } from './context/local.js';
import { ConcurrentOutput } from '../../private/node/ui/components/ConcurrentOutput.js';
import { render, renderOnce } from '../../private/node/ui.js';
import { alert } from '../../private/node/ui/alert.js';
import { FatalError } from '../../private/node/ui/components/FatalError.js';
import { Table } from '../../private/node/ui/components/Table/Table.js';
import { SelectPrompt } from '../../private/node/ui/components/SelectPrompt.js';
import { Tasks } from '../../private/node/ui/components/Tasks.js';
import { TextPrompt } from '../../private/node/ui/components/TextPrompt.js';
import { AutocompletePrompt } from '../../private/node/ui/components/AutocompletePrompt.js';
import React from 'react';
import { AbortController } from '@shopify/cli-kit/node/abort';
/**
 * Renders output from concurrent processes to the terminal with {@link ConcurrentOutput}.
 */
export async function renderConcurrent({ renderOptions = {}, ...props }) {
    const newProps = {
        abortController: new AbortController(),
        ...props,
    };
    return render(React.createElement(ConcurrentOutput, { ...newProps }), {
        ...renderOptions,
        exitOnCtrlC: typeof props.onInput === 'undefined',
    });
}
/**
 * Renders an information banner to the console.
 *
 * Basic:
 *
 * ```
 * ╭─ info ───────────────────────────────────────────────────╮
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 *
 * Complete:
 * ```
 * ╭─ info ───────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * │  Next steps                                              │
 * │    • Run `cd santorini-goods`                            │
 * │    • To preview your project, run `npm app dev`          │
 * │    • To add extensions, run `npm generate extension`     │
 * │                                                          │
 * │  Reference                                               │
 * │    • Run `npm shopify help`                              │
 * │    • Press 'return' to open the dev docs:                │
 * │      https://shopify.dev                                 │
 * │                                                          │
 * │  Link: https://shopify.com                               │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export function renderInfo(options) {
    return alert({ ...options, type: 'info' });
}
/**
 * Renders a success banner to the console.
 *
 * Basic:
 *
 * ```
 * ╭─ success ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 *
 * Complete:
 * ```
 * ╭─ success ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * │  Next steps                                              │
 * │    • Run `cd santorini-goods`                              │
 * │    • To preview your project, run `npm app dev`            │
 * │    • To add extensions, run `npm generate extension`       │
 * │                                                          │
 * │  Reference                                               │
 * │    • Run `npm shopify help`                                │
 * │    • Press 'return' to open the dev docs:                │
 * │      https://shopify.dev                                 │
 * │                                                          │
 * │  Link: https://shopify.com                               │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export function renderSuccess(options) {
    return alert({ ...options, type: 'success' });
}
/**
 * Renders a warning banner to the console.
 *
 * Basic:
 *
 * ```
 * ╭─ warning ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 *
 * Complete:
 * ```
 * ╭─ warning ────────────────────────────────────────────────╮
 * │                                                          │
 * │  Title                                                   │
 * │                                                          │
 * │  Body                                                    │
 * │                                                          │
 * │  Next steps                                              │
 * │    • Run `cd santorini-goods`                            │
 * │    • To preview your project, run `npm app dev`          │
 * │    • To add extensions, run `npm generate extension`     │
 * │                                                          │
 * │  Reference                                               │
 * │    • Run `npm shopify help`                              │
 * │    • Press 'return' to open the dev docs:                │
 * │      https://shopify.dev                                 │
 * │                                                          │
 * │  Link: https://shopify.com                               │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export function renderWarning(options) {
    return alert({ ...options, type: 'warning' });
}
/**
 * Renders a Fatal error to the console inside a banner.
 *
 * ```
 * ╭─ error ──────────────────────────────────────────────────╮
 * │                                                          │
 * │  Couldn't connect to the Shopify Partner Dashboard.      │
 * │                                                          │
 * │  Check your internet connection and try again.           │
 * │                                                          │
 * ╰──────────────────────────────────────────────────────────╯
 * ```
 */
export function renderFatalError(error) {
    return renderOnce(React.createElement(FatalError, { error: error }), 'error', consoleError);
}
/**
 * Renders a select prompt to the console.
 *
 * ```
 * ?  Associate your project with the org Castile Ventures?
 *
 *      Add:     • new-ext
 *
 *      Remove:  • integrated-demand-ext
 *               • order-discount
 *
 * \>  (f) first
 *     (s) second
 *     (3) third
 *     (4) fourth
 *     (5) seventh
 *     (6) tenth
 *
 *     Automations
 *     (7) fifth
 *     (8) sixth
 *
 *     Merchant Admin
 *     (9) eighth
 *     (10) ninth
 *
 *     Press ↑↓ arrows to select, enter to confirm
 * ```
 */
export function renderSelectPrompt(props) {
    // eslint-disable-next-line max-params
    return new Promise((resolve, reject) => {
        render(React.createElement(SelectPrompt, { ...props, onSubmit: (value) => resolve(value) }), {
            exitOnCtrlC: false,
        }).catch(reject);
    });
}
/**
 * Renders a confirmation prompt to the console.
 *
 * ?  Do you want to continue?
 *
 * \>  (y) Yes, confirm
 *     (n) No, canccel
 *
 *     Press ↑↓ arrows to select, enter to confirm
 */
export function renderConfirmationPrompt({ message, infoTable, confirmationMessage = 'Yes, confirm', cancellationMessage = 'No, cancel', }) {
    const choices = [
        {
            label: confirmationMessage,
            value: true,
            key: 'y',
        },
        {
            label: cancellationMessage,
            value: false,
            key: 'n',
        },
    ];
    return renderSelectPrompt({
        choices,
        message,
        infoTable,
        submitWithShortcuts: true,
    });
}
/**
 * Renders an autocomplete prompt to the console.
 * ```
 * ?  Select a template  Type to search...

 * \>  first
 *     second
 *     third

 *  Press ↑↓ arrows to select, enter to confirm
 * ```
 */
export function renderAutocompletePrompt(props) {
    const newProps = {
        search(term) {
            return Promise.resolve({
                data: props.choices.filter((item) => item.label.toLowerCase().includes(term.toLowerCase())),
            });
        },
        ...props,
    };
    // eslint-disable-next-line max-params
    return new Promise((resolve, reject) => {
        render(React.createElement(AutocompletePrompt, { ...newProps, onSubmit: (value) => resolve(value) }), {
            exitOnCtrlC: false,
        }).catch(reject);
    });
}
/**
 * Renders a table to the console.
 *
 * ```
 * name                      role           Identifier
 * ────────────────────────  ─────────────  ──────────
 * Dawn                      [live]         #1361
 * Studio                                   #1363
 * Debut                     [unpublished]  #1374
 * Development (1a23b4-MBP)  [development]  #1368
 * ```
 */
export function renderTable(props) {
    return renderOnce(React.createElement(Table, { ...props }));
}
/**
 * Runs async tasks and displays their progress to the console.
 */
export async function renderTasks(tasks) {
    return render(React.createElement(Tasks, { tasks: tasks }));
}
/**
 * Renders a text prompt to the console.
 * ```
 * ?  What is your name?
 * \>  John
 * ```
 */
export function renderTextPrompt(props) {
    // eslint-disable-next-line max-params
    return new Promise((resolve, reject) => {
        render(React.createElement(TextPrompt, { ...props, onSubmit: (value) => resolve(value) }), {
            exitOnCtrlC: false,
        }).catch(reject);
    });
}
/** Renders a text string to the console.
 * Using this function makes sure that correct spacing is applied among the various components. */
export function renderText({ text, logLevel = 'info', logger = consoleLog }) {
    let textWithLineReturn = text;
    if (!text.endsWith('\n'))
        textWithLineReturn += '\n';
    if (isUnitTest())
        collectLog(logLevel, textWithLineReturn);
    outputWhereAppropriate(logLevel, logger, textWithLineReturn);
}
/** Waits for any key to be pressed except Ctrl+C which will terminate the process. */
export const keypress = async () => {
    // eslint-disable-next-line max-params
    return new Promise((resolve, reject) => {
        const handler = (buffer) => {
            process.stdin.setRawMode(false);
            process.stdin.pause();
            const bytes = Array.from(buffer);
            if (bytes.length && bytes[0] === 3) {
                outputDebug('Canceled keypress, User pressed CTRL+C');
                reject(new AbortSilentError());
            }
            process.nextTick(resolve);
        };
        process.stdin.resume();
        process.stdin.setRawMode(true);
        process.stdin.once('data', handler);
    });
};
//# sourceMappingURL=ui.js.map