import {
  __spreadProps,
  __spreadValues
} from "./chunk-JUWXSDKJ.js";
import { spyOn } from "tinyspy";
import { HeadersMock } from "./headers.js";
import { FetchMockInstance } from "./mock.js";
import { ResponseMock } from "./response.js";
const methods = ["GET", "POST", "PATCH", "PUT", "DELETE"];
class MockStorage {
  storage = [];
  will(method, type, path, includesQuery, once, value) {
    this.storage.push({
      method,
      path,
      once,
      result: { type, value },
      includesQuery
    });
  }
  getApiCall(fetchMethod, urlPath) {
    const storeIndex = this.storage.findIndex(({ path, method, includesQuery }) => {
      if (fetchMethod !== method)
        return false;
      const fetchPath = !includesQuery && typeof path === "string" ? urlPath.split("?")[0] : urlPath;
      if (typeof path === "string" && path === fetchPath) {
        return true;
      } else if (path instanceof RegExp && path.test(fetchPath)) {
        return true;
      }
      return false;
    });
    if (storeIndex === -1) {
      return void 0;
    }
    const store = this.storage[storeIndex];
    if (store.once) {
      this.storage.splice(storeIndex, 1);
    }
    return store.result;
  }
  clear(fetchMethod, fetchUrl) {
    this.storage = this.storage.filter(({ method, path }) => {
      return method !== fetchMethod && path !== fetchUrl;
    });
  }
  clearAll() {
    this.storage = [];
  }
}
const Mocks = new MockStorage();
const settings = {
  global: globalThis,
  fetchKey: "fetch"
};
async function getDataFromMockResult(result, url, [input, init]) {
  const defaults = {
    body: {},
    statusCode: 200,
    statusText: "OK",
    headers: [["Content-Type", "application/json"]]
  };
  if (typeof result.value !== "function") {
    return __spreadValues(__spreadValues({}, defaults), result.value);
  }
  const headers = result.value.headers;
  const data = await result.value(new URL(url), input, init);
  return __spreadValues(__spreadProps(__spreadValues({}, defaults), {
    headers
  }), data);
}
function prepareFetch(obj = globalThis, key = "fetch") {
  settings.global = obj;
  settings.fetchKey = key;
  const originalFetch = obj[key] || ((path) => {
    const url = typeof path === "string" ? path : path.url;
    throw new Error(`${key} is not defined. tried fetching "${url}"`);
  });
  obj[key] = async (urlOrRequest, optionsOrNothing) => {
    const url = typeof urlOrRequest === "string" ? urlOrRequest : urlOrRequest.url;
    const options = typeof optionsOrNothing !== "undefined" ? optionsOrNothing : { method: "GET" };
    const method = typeof urlOrRequest !== "string" ? urlOrRequest.method : options.method || "GET";
    let result = Mocks.getApiCall(method.toUpperCase(), url);
    if (result === void 0) {
      return originalFetch(urlOrRequest, optionsOrNothing);
    }
    if (result.type === "throws") {
      throw result.value;
    }
    const { body, statusCode, statusText, headers } = await getDataFromMockResult(result, url, [
      urlOrRequest,
      optionsOrNothing
    ]);
    return new ResponseMock(url, body, {
      status: statusCode,
      statusText,
      headers: new HeadersMock(headers)
    });
  };
}
function spyOnFetch(fetchMethod, fetchPath, includeQuery = true) {
  const spyFetch = spyOn(settings.global, settings.fetchKey);
  if (!(fetchPath instanceof RegExp)) {
    fetchPath = this.options.baseUrl + fetchPath;
  }
  return new FetchMockInstance({
    method: fetchMethod,
    url: fetchPath,
    includeQuery,
    baseUrl: this.options.baseUrl,
    spy: spyFetch,
    headers: new HeadersMock()
  });
}
function setGlobals(globals) {
  Object.assign(settings, globals);
}
spyOnFetch.clearAll = Mocks.clearAll.bind(Mocks);
spyOnFetch.options = {
  baseUrl: "",
  get global() {
    return settings.global;
  },
  get fetchKey() {
    return settings.fetchKey;
  }
};
spyOnFetch.setOptions = (options) => {
  if (typeof options.baseUrl === "string") {
    spyOnFetch.options.baseUrl = options.baseUrl;
  }
  setGlobals(options);
};
const mockFetch = spyOnFetch.bind(spyOnFetch);
mockFetch.clearAll = spyOnFetch.clearAll;
mockFetch.options = spyOnFetch.options;
mockFetch.setOptions = spyOnFetch.setOptions;
const createAlias = (method) => spyOnFetch.bind(spyOnFetch, method);
const mockGet = createAlias("GET");
const mockPost = createAlias("POST");
const mockPatch = createAlias("PATCH");
const mockDelete = createAlias("DELETE");
const mockPut = createAlias("PUT");
function createMockFetch({
  baseUrl = "",
  global = settings.global,
  fetchKey = settings.fetchKey
} = {}) {
  const options = { baseUrl, global, fetchKey };
  const mockFetch2 = spyOnFetch.bind({ options });
  mockFetch2.options = options;
  mockFetch2.clearAll = spyOnFetch.clearAll;
  mockFetch2.setOptions = (opts) => {
    if (typeof options.baseUrl === "string") {
      mockFetch2.options.baseUrl = options.baseUrl;
    }
    setGlobals(opts);
  };
  const createAlias2 = (method) => {
    return mockFetch2.bind(mockFetch2, method);
  };
  return {
    mockFetch: mockFetch2,
    mockGet: createAlias2("GET"),
    mockPost: createAlias2("POST"),
    mockPatch: createAlias2("PATCH"),
    mockDelete: createAlias2("DELETE"),
    mockPut: createAlias2("PUT")
  };
}
export {
  Mocks,
  createMockFetch,
  mockDelete,
  mockFetch,
  mockGet,
  mockPatch,
  mockPost,
  mockPut,
  prepareFetch
};
